;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;; $Source: /home/geddis/archive/cvs/scripts/dtp/lisp/view.lisp,v $
;;; $Id: view.lisp,v 1.1 2004/07/15 04:22:05 geddis Exp $
;;;
;;; (c) Copyright 1994-2004 Don Geddis.  All rights reserved.
;;;
;;; System:		Don's Theorem Prover (DTP)
;;; Written by:		Don Geddis <don@geddis.org>
;;;
;;;----------------------------------------------------------------------------

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(show) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defgeneric show (object))

(defmethod show ((object proof))
  (let ((*proof* object))
    (show-toplevel object) ))

(defmethod show ((object answer))
  (let ((*proof* (answer-proof object)))
    (show-toplevel object) ))

;;;----------------------------------------------------------------------------

(defun make-proof-figure (proof-or-answer &optional (outfile nil))
  (if outfile
      (when *figure-directory*
	(setq outfile (concatenate 'string *figure-directory* outfile))
	(setq outfile (translate-logical-pathname outfile)) )
    (setq outfile *ps-file*) )
  (setq *explored* nil)
  (setq *id-assigned* nil)
  (setq *unique-id-assigned* nil)
  (with-open-file (f *dot-file* :direction :output :if-exists :supersede)
    (if *debug-view*
	(setq *show-stream* *standard-output*)
      (setq *show-stream* f) )
    (let ((*print-pretty* nil))
      (dot-header proof-or-answer)
      (output proof-or-answer)
      (dot-footer proof-or-answer) ))
  (unless *debug-view*
    (when (probe-file outfile)
      (delete-file outfile) )
    #+cmu
    (ext:run-program *dot-command* `("-Tps" ,*dot-file*))
    #+allegro
    (excl:run-shell-command
     (format nil "~A -Tps ~A > ~A" *dot-command* *dot-file* outfile) )
    ))

;;;----------------------------------------------------------------------------
;;;
;;;	Private

;;;----------------------------------------------------------------------------

(defun show-toplevel (object)
  (setq *explored* nil)
  (setq *id-assigned* nil)
  (setq *unique-id-assigned* nil)
  (if *graphic-display*
      (progn
	(with-open-file (f *dot-file* :direction :output :if-exists :supersede)
	  (if *debug-view*
	      (setq *show-stream* *standard-output*)
	    (setq *show-stream* f) )
	  (let ((*print-pretty* nil))
	    (dot-header object)
	    (output object)
	    (dot-footer object) ))
	(unless *debug-view*
	  (when (probe-file *ps-file*) (delete-file *ps-file*))
	  #+cmu
	  (progn
	    (ext:run-program *dot-command* `("-Tps" ,*dot-file*))
	    (if (and *display-landscape* (not *display-as-figure*))
		(ext:run-program
		 *ps-viewer-command* `("-landscape" ,*ps-file*) )
	      (ext:run-program *ps-viewer-command* `(,*ps-file*) :wait nil)
	      ))
	  #+allegro
	  (progn
	    (excl:run-shell-command
	     (format nil "~A -Tps ~A > ~A" *dot-command* *dot-file* *ps-file*)
	     )
	    (excl:run-shell-command
	     (format nil "~A~A ~A"
		     *ps-viewer-command*
		     (if (and *display-landscape* (not *display-as-figure*))
			 " -landscape" "" )
		     *ps-file* )
	     :wait nil ))
	  ))
    (progn
      (setq *show-stream* *standard-output*)
      (let ((*depth* 0))
	(output object) ))))

;;;----------------------------------------------------------------------------

(defun output (object &rest rest &key (answer nil) &allow-other-keys)
  (if answer
      (let ((my-id (dot-id-of object :unique t)))
	(when *debug-view* (push (cons object answer) *all-answers*))
	(if *graphic-display*
	    (apply #'dotify object :my-id my-id rest)
	  (apply #'textify object :my-id my-id rest) )
	(apply #'show-below object :my-id my-id rest) )
    (progn
      (if *graphic-display*
	  (apply #'dotify object rest)
	(apply #'textify object rest) )
      (apply #'show-below object rest) )))

;;;----------------------------------------------------------------------------
;;;
;;;	Dot headers and footers

(defgeneric dot-header (object))

(defun general-dot-header (&key (proof nil) (answer nil))
  (format *show-stream* "/* DTP ~A */~%" *dtp-version*)
  (when proof
    (format *show-stream* "/* Proof of ~A */~%" (proof-query proof)) )
  (when answer
    (format *show-stream* "/* Proof of ~A */~%" (apply-answer answer)) )
  (format *show-stream* "digraph proof {~%")
  (format *show-stream* "  page = \"8.5,11\" ;~%")
  (format *show-stream* "  ordering = out ;~%")
  (cond
   (*display-as-figure*
    (format *show-stream* "  margin = \"0,0\" ;~%")
    (format *show-stream* "  size = \"~A\" ;~%" *figure-size*)
    (format *show-stream* "  ratio = compress ;~%")
    )
   (*display-one-page*
    (format *show-stream* "  margin = \".25,.25\" ;~%")
    (format *show-stream* "  size = \"8,10.5\" ;~%")
    (if (eq *display-one-page* :auto)
	(format *show-stream* "  ratio = auto ;~%")
      (format *show-stream* "  ratio = compress ;~%") )
    (format *show-stream* "  center = true ;~%") ))
  (when *display-landscape*
    (if *display-as-figure*
	(format *show-stream* "  rotate = 90 ;~%")
      (format *show-stream* "  orientation = landscape ;~%") ))
  (format *show-stream* "  ranksep = .2 ;~%")
  (format *show-stream* "  nodesep = .1 ;~%")
  (format *show-stream* "  fontsize = 14 ;~%")
  (format *show-stream* "  node [fontsize=11,height=0,width=0] ;~%")
  (format *show-stream* "  edge [fontsize=9,weight=5] ;~%")
  )

(defmethod dot-header ((object proof))
  (when *debug-view* (setq *all-answers* nil))
  (general-dot-header :proof object)
  (when (and *display-title* (not *display-as-figure*))
    (format *show-stream* "  label = \"\\nProof space of ~:(~A~)\\n"
	    (proof-query object) )
    (format *show-stream* "Theory ~:(~A~), ~D answer~:P, "
	    (proof-theory object) (length (proof-answers object)) )
    (if (or (active-agenda object)
	    (proof-query-conjunctions object) )
	(format *show-stream* "In progress")
      (format *show-stream* "Complete") )
    (format *show-stream* "\" ;~%") ))

(defmethod dot-header ((object answer))
  (when *debug-view* (setq *all-answers* nil))
  (general-dot-header :answer object)
  (when (and *display-title* (not *display-as-figure*))
    (let ((*print-pretty* nil))
      (format *show-stream* "  label = \"\\nProof of ~:(~A~)\" ;~%"
	      (apply-answer object) )
      )))

(defgeneric dot-footer (object))

(defmethod dot-footer ((object proof))
  (when *debug-view* (setq *all-answers* (reverse *all-answers*)))
  (format *show-stream* "}~%") )

(defmethod dot-footer ((object answer))
  (when *debug-view* (setq *all-answers* (reverse *all-answers*)))
  (format *show-stream* "}~%") )

;;;----------------------------------------------------------------------------
;;;
;;;	Dot-Name-Of

(defgeneric dot-name-of (object &optional answer))

;;;----------------------------------------------------------------------------

(defmethod dot-name-of ((object proof) &optional (answer nil))
  (declare (ignore answer))
  (format nil "~:(~A~)" (proof-query object)) )

;;;----------------------------------------------------------------------------

(defmethod dot-name-of ((object answer) &optional (answer nil))
  (declare (ignore answer))
  (format nil "~:(~A~)" (apply-answer object)) )

;;;----------------------------------------------------------------------------

(defmethod dot-name-of ((object dtp-subgoal) &optional (answer nil))
  (with-output-to-string (s)
    (if (typep answer 'answer)
	(loop
	    with lit = (slot-value object 'literal)
	    for bl in (answer-ae-binding-lists answer)
	    collect (literal-plug lit bl) into lits
	    finally
	      (setq lits
		(cons (literal-plug lit (answer-binding-list answer)) lits) )
	      (setq lits (remove-duplicates lits :test #'literal-equal-p))
	      (loop
		  for each-lit in lits
		  for between = nil then t
		  do (when between (format s " or "))
		     (print-literal-node each-lit :s s) ))
      (print-literal-node (slot-value object 'literal) :s s) )))

;;;----------------------------------------------------------------------------

(defmethod dot-name-of ((object dtp-conjunction) &optional (answer nil))
  (cond
   ((slot-value object 'list)
    (with-output-to-string (s)
      (loop
	  with bl = (when (typep answer 'answer)
		      (answer-binding-list answer) )
	  with conjuncts = (slot-value object 'list)
	  for count from 0
	  for virtual =
	    (and (typep object 'dtp-forked-conjunction)
		 (< count (slot-value object 'top-conjunct)) )
	  for conjunct = (pop conjuncts)
	  while conjunct
	  do (format s "<c~D> " count)
	     (when virtual (format s "["))
	     (print-literal-node
	      (if bl
		  (literal-plug (slot-value conjunct 'literal) bl)
		(slot-value conjunct 'literal) )
	      :s s )
	     (when virtual (format s "]"))
	     (when conjuncts (format s "|")) )))
   ((and (list-of-length-one-p (slot-value object 'answers))
	 (typep (answer-justification (first (slot-value object 'answers)))
		's-cache-justification ))
    :cache )
   (t
    :lookup )))

;;;----------------------------------------------------------------------------

(defmethod dot-name-of ((object dtp-conjunct) &optional (answer nil))
  (declare (ignore answer))
  (with-output-to-string (s)
    (print-literal-node (slot-value object 'literal) :s s) ))

;;;----------------------------------------------------------------------------

(defun dot-id-of (object &key (unique nil))
  "Return integer ID...if UNIQUE, then every new call results in new ID"
  (if unique
      (progn
	(add-to-end object *unique-id-assigned*)
	(- (length *unique-id-assigned*)) )
    (progn
      (add-to-end-if-new object *id-assigned*)
      (position object *id-assigned*) )))

(defun dot-id-to-object (id)
  (if (>= id 0)
      (nth id *id-assigned*)
    (nth (1- (- id)) *unique-id-assigned*) ))

(defun dot-node-name (id-or-object)
  (let ((id id-or-object))
    (unless (typep id-or-object 'integer)
      (setq id (dot-id-of id-or-object)) )
    (if (>= id 0)
	(format nil "n~D" id)
      (format nil "nu~D" (- id)) )))

;;;----------------------------------------------------------------------------

(defmethod binding-label-between
    ((subgoal dtp-subgoal)
     (conjunction dtp-conjunction)
     &optional (answer nil) )
  (let (binding-list answer-vars)
    (setq binding-list (bl-from-inference subgoal conjunction))
    (when answer
      (setq answer-vars
	(mapcar #'binding-variable (answer-binding-list answer)) )
      (setq binding-list
	(remove-if #'(lambda (var) (find var answer-vars))
		   binding-list :key #'binding-variable )))
    (if binding-list
	(dot-name-of-bl binding-list)
      (when (setq binding-list (slot-value conjunction 'ae-binding-list))
	(dot-name-of-bl binding-list :disjunction t) ))
    ))

(defun bl-from-inference (subgoal conjunction)
  "Return binding from inference which affects subgoal"
  (let ((good-vars (literal-vars-in (slot-value subgoal 'literal))))
    (remove-if-not
     #'(lambda (binding)
	 (find (binding-variable binding) good-vars) )
     (slot-value conjunction 'binding-list) )))

(defun dot-name-of-bl (binding-list &key (disjunction nil))
  "Takes a binding list, returns a DOT label (string)"
  (with-output-to-string (s)
    (loop
	for binding-pair = (pop binding-list)
	while binding-pair
	unless (eq (binding-variable binding-pair) 't)
	do (when disjunction (format s "Or "))
	   (format s "~(~A~)->"
		   (variable-to-string (binding-variable binding-pair)) )
	   (term-to-string (binding-value binding-pair) s)
	   (when binding-list
	     (format s "\\n") ))
    ))

;;;----------------------------------------------------------------------------

(defmethod binding-label-between
    ((subgoal dtp-subgoal) (r-just r-justification) &optional (answer nil))
  (let ((an-sg (r-just-ancestor-subgoal r-just))
	(lf-sg (r-just-leaf-subgoal r-just))
	bl )
    (setq bl
      (dtp-ify-binding-list
       (literal-negated-pair-p
	(slot-value lf-sg 'literal) (slot-value an-sg 'literal) )))
    (when answer
      (let ((answer-vars
	     (mapcar #'binding-variable (answer-binding-list answer)) ))
	(setq bl
	  (remove-if #'(lambda (var) (find var answer-vars))
		     bl :key #'binding-variable ))))
    (when bl (dot-name-of-bl bl)) ))

;;;----------------------------------------------------------------------------

(defun hidden-conjunction? (conjunction)
  (or (list-of-length-one-p (slot-value conjunction 'list))
      (and (typep conjunction 'dtp-forked-conjunction)
	   (not *display-blocked-separately*) )))

(defun hidden-subgoal?
    (subgoal answer parent-id record previous-sg-hidden previous-sg)
  "Hide subgoal if parent is a displayed conjunction with same conjunct"
  (when parent-id
    (let ((parent (dot-id-to-object parent-id)))
      (and (typep parent 'dtp-conjunction)
	   (not previous-sg-hidden)
	   (or answer
	       (let ((pc (parent-conjuncts subgoal)))
		 (and (or (list-of-length-one-p pc)
			  (and (eq *display-cache-links* :nodes)
			       (not (slave-link?
				     subgoal answer previous-sg ))))
		      (literal-equal-p
		       (slot-value subgoal 'literal)
		       (slot-value
			(nth record (slot-value parent 'list))
			'literal ))))
	       )))))

(defun parent-conjuncts (subgoal)
  (let (pc)
    (setq pc
      (remove-duplicates
       (append (slot-value subgoal 'conjuncts-to-propagate-to)
	       (slot-value subgoal 'used-conjuncts) )))
    (unless *display-blocked-separately*
      (setq pc
	(remove-if
	 #'(lambda (c)
	     (typep (slot-value c 'parent-conjunction)
		    'dtp-forked-conjunction ))
	 pc )))
    pc ))

(defun slave-link? (subgoal answer previous-sg)
  "True IFF the link between PREVIOUS-SG and SUBGOAL is a slave link"
  (not (or answer
	   (eq previous-sg (slot-value subgoal 'parent-subgoal)) )))

;;;----------------------------------------------------------------------------

(defun display-query-p (proof)
  "True IFF the graph should have a seperate query node"
  (and (not *display-as-figure*)
       (or (eq *display-query* t)
	   (and (eq *display-query* :auto)
		(not (= (length (objects-below proof)) 1) )))))

;;;----------------------------------------------------------------------------
