;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;; $Source: /home/geddis/archive/cvs/scripts/dtp/lisp/variables.lisp,v $
;;; $Id: variables.lisp,v 1.1 2004/07/15 04:22:05 geddis Exp $
;;;
;;; (c) Copyright 1994-2004 Don Geddis.  All rights reserved.
;;;
;;; System:		Don's Theorem Prover (DTP)
;;; Written by:		Don Geddis <don@geddis.org>
;;;
;;;----------------------------------------------------------------------------

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(possible-values initial-value) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Parameters

(defparameter *dtp-package* (find-package "DTP"))

(defparameter *test-ignore-answer* '**)

(defparameter *figure-size* "6,8")
(defparameter *thesis-figure-directory* "dtp:figures;")
(defparameter *manual-figure-directory* "dtp:figures;")

(defparameter *timeout-resolution* 25
  "Number of loop iterations before checking elapsed time" )
(defparameter *tptp-timeout* 60 "Timeout in sixty seconds")

;; TPTP
(defparameter *directory-separator* #-mcl "/" #+mcl ":")
(defparameter *tptp-library* "dtp:tptp;")
(defparameter *tptp-problems* "problems;")
(defparameter *tptp-axioms* "axioms;")
(defparameter *tptp-kif* "kif;")
(defparameter *tptp-report-file*
    #+unix "/tmp/tptp.report" #-unix "tptp.report" )

;; Graphic Display
(defparameter *dot-command* "dot")
(defparameter *dot-file* "/tmp/proof.dot")
(defparameter *ps-viewer-command* "ghostview")
(defparameter *ps-file* "/tmp/proof.ps")
(defparameter *dash-ps-style*
    "\".5 setlinewidth [3 InvScaleFactor mul dup] 0 setdash\""
  "`dashed', `dotted', or quoted Postscript quote" )
(defparameter *terminal-node-dot-type*
    (concatenate 'string
      "shape=record,fontsize=7,style=filled,"
      "color=black,fontcolor=white,fontname=Helvetica" )
  "Type of DOT node to use for Reductions, Cache, Lookup, etc. nodes" )

;;;----------------------------------------------------------------------------
;;;
;;;	Initialization (reset restores these values)

(defparameter *dtp-possible-values*
    '(
      (*initial-subgoal-depth* .
       "Nil (for unbounded), or a non-negative integer" )
      (*subgoal-depth-skip* .
       "Nil (for unbounded), or a non-negative integer" )
      (*subgoal-maximum-depth* .
       "Nil (for unbounded), or a non-negative integer" )
      (*initial-function-depth* .
       "Nil (for unbounded), or a non-negative integer" )
      (*function-depth-skip* .
       "Nil (for unbounded), or a non-negative integer" )
      (*function-maximum-depth* .
       "Nil (for unbounded), or a non-negative integer" )
      (*caching* .
       (nil :iap :success :failure :answers :subgoals :recursion
	:postponement ))
      (*cache-size* . "Nil (for unbounded), or a non-negative integer")
      (*search-order* . (:dfs :bfs))
      (*trace* .
       (:file-load :tests :answers :subgoals :proofs :conjunct-proofs
	:backtrack :cutoffs :timeout :iteration :caching :solutions ))
      (*print-variables-specially* . (nil :no-prefix :with-renaming))
      (*display-one-page* . (nil :auto t))
      (*display-query* . (nil :auto t))
      (*display-cache-links* . (nil :nodes t))
      ))

(defun possible-values (variable)
  (rest (assoc variable *dtp-possible-values*)) )

(defparameter *dtp-initial-values*
    '(
      (*theory* . global)
      (*last-proof* . nil)
      
      ;; Inference Options
      (*use-negated-goal* . t)
      (*use-subgoal-inference* . t)
      (*use-contrapositives* . t)
      (*use-reduction* . t)
      (*use-pure-literal-elimination* . t)
      (*use-backjumping* . t)
      (*use-residue* . t)
      (*use-procedural-attachments* . t)
      (*use-reordering* . t)
      (*timeout-maximum-seconds* . nil)
      (*search-order* . :dfs)

      ;; Caching
      (*caching* . :recursion)
      (*cache-reductions* . t)
      (*use-unblocking* . t)
      (*cache-size* . nil)	; Unbounded cache

      ;; Iteration
      (*use-subgoal-cutoffs* . nil)
      (*initial-subgoal-depth* . nil)
      (*subgoal-depth-skip* . nil)
      (*subgoal-maximum-depth* . nil)
      (*use-function-cutoffs* . nil)
      (*initial-function-depth* . nil)
      (*function-depth-skip* . nil)
      (*function-maximum-depth* . nil)

      ;; Residue
      (*assumables* . nil)
      (*consistency-check* . nil)
      
      ;; Tracing
      (*trace* . (:file-load :tests))
      (*single-step* . nil)
      ))

(defparameter *dtp-internal-initial-values*
    '(
      (*node-id-count* . 0)
      (*all-gensymed-variables* . nil)

      ;; View
      (*show-stream* . nil)
      (*explored* . nil)
      (*id-assigned* . nil)
      (*unique-id-assigned* . nil)
      (*debug-view* . nil)
      (*all-answers* . nil)
      ))

(defun initial-value (variable)
  (or
   (rest (assoc variable *dtp-initial-values*))
   (rest (assoc variable *dtp-internal-initial-values*)) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Variables

;; Display form
(defvar *print-logic-as-lists* nil)
(defvar *print-variables-specially* nil)
(defvar *graphic-display*
  #-mcl t #+mcl nil                     ; Requires AT&T's "dot" [unix only]
  "Toggle: Output in Postscript [T] or text?" )

;; Misc
(defvar *theory*)
(defvar *last-proof*)

;; Inference engine
(defvar *use-negated-goal*)
(defvar *use-subgoal-inference*)
(defvar *use-contrapositives*)
(defvar *use-reduction*)
(defvar *use-pure-literal-elimination*)
(defvar *use-backjumping*)
(defvar *use-residue*)
(defvar *use-procedural-attachments*)
(defvar *use-reordering*)
(defvar *timeout-maximum-seconds*)
(defvar *search-order*)

;; Caching
(defvar *caching*)
(defvar *cache-reductions*)
(defvar *use-unblocking* t "Only matters for :recursion/:postponement caching")
(defvar *cache-size* nil "Nil (for unbounded) or positive integer")

;; Iteration
(defvar *use-subgoal-cutoffs*)
(defvar *initial-subgoal-depth*)
(defvar *subgoal-depth-skip*)
(defvar *subgoal-maximum-depth*)
(defvar *use-function-cutoffs*)
(defvar *initial-function-depth*)
(defvar *function-depth-skip*)
(defvar *function-maximum-depth*)

;; Residue
(defvar *assumables*)
(defvar *consistency-check*)

;; Tracing
(defvar *trace*)
(defvar *proof-line-count*)
(defvar *depth*)
(defvar *subgoal-map*)
(defvar *single-step*)

;; Graphic Display
(defvar *display-one-page* :auto "Centering -> true, nil/:auto/t")
(defvar *display-as-figure* nil "Margins -> 0")
(defvar *display-landscape* t "Orientation")
(defvar *display-title* t "Label at bottom")
(defvar *display-query* :auto "Single root node, nil/:auto/t")
(defvar *display-color* nil)
(defvar *display-cache-links* t "nil/:nodes/t")
(defvar *display-slaved-at-end?* t)
(defvar *display-blocked-separately* nil)

;; Internal
(defvar *node-id-count*)
(defvar *all-gensymed-variables* nil)	; So we can unintern them on reset
(defvar *proof*)			; Dynamic variable, set to (make-proof)
(defvar *goal-node-id-count*)
(defvar *timeout-count*)
(defvar *timeout-end*)
(defvar *figure-directory*)

;; Internal: View
(defvar *show-stream* nil "Output on file (for dot) or term (for text)")
(defvar *explored* nil "If we've seen this node before, don't explore below")
(defvar *id-assigned* nil "List of objects (id = position in list)")
(defvar *unique-id-assigned* nil "List of objects (negative id = pos in list)")
(defvar *debug-view* nil "Sends DOT output to *standard-io* instead")
(defvar *all-answers* nil)

;;;----------------------------------------------------------------------------

(defparameter *user-setable-variables*
    `(("General"
       *theory* *assumables* *consistency-check* )
      ("Inference Engine"
       *use-negated-goal* *use-subgoal-inference* *use-contrapositives*
       *use-reduction* *use-pure-literal-elimination* *use-backjumping*
       *use-residue* *use-procedural-attachments* *use-reordering*
       *timeout-maximum-seconds* *search-order* )
      ("Caching"
       *caching* *cache-reductions* *use-unblocking* *cache-size* )
      ("Iteration"
       *use-subgoal-cutoffs* *initial-subgoal-depth* *subgoal-depth-skip*
       *subgoal-maximum-depth* *use-function-cutoffs* *initial-function-depth*
       *function-depth-skip* *function-maximum-depth* )
      ("Printed form"
       *print-logic-as-lists* *print-variables-specially* *graphic-display* )
      ("Graphic Display"
       *display-one-page* *display-as-figure* *display-landscape*
       *display-title* *display-query* *display-color* *display-cache-links*
       *display-slaved-at-end?* *display-blocked-separately* )
      ("Tracing"
       *trace* *single-step* )
      ))

;;;----------------------------------------------------------------------------

(defun reset-variables
    (&key (unless-set nil) (only-internal nil) (only-user nil))
  (cond
   (unless-set
    (dolist (pair (append *dtp-initial-values* *dtp-internal-initial-values*))
      (unless (boundp (car pair))
	(set (car pair) (cdr pair)) )))
   (only-internal
    (let ((users (apply #'append (mapcar #'cdr *user-setable-variables*))))
      (dolist
	  (pair (append *dtp-initial-values* *dtp-internal-initial-values*))
	(unless (find (car pair) users)
	  (set (car pair) (cdr pair)) ))))
   (only-user
    (dolist (var (apply #'append (mapcar #'cdr *user-setable-variables*)))
      (set var (initial-value var)) ))
   (t
    (dolist (pair (append *dtp-initial-values* *dtp-internal-initial-values*))
      (set (car pair) (cdr pair)) ))))

(eval-when (load eval)
  (reset-variables :unless-set t) )

;;;----------------------------------------------------------------------------
