;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;; $Source: /home/geddis/archive/cvs/scripts/dtp/lisp/test.lisp,v $
;;; $Id: test.lisp,v 1.1 2004/07/15 04:22:05 geddis Exp $
;;;
;;; (c) Copyright 1994-2004 Don Geddis.  All rights reserved.
;;;
;;; System:		Don's Theorem Prover (DTP)
;;; Written by:		Don Geddis <don@geddis.org>
;;;
;;;----------------------------------------------------------------------------

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(test-dtp load-logic-samples test-tptp tptp-stats) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defun test-dtp (&key (reset t))
  "Resets theorem prover, loads and runs test suite"
  (when reset
    (reset-dtp)
    (load-logic-samples) )
  (let (testfile)
    (setq testfile (concatenate 'string *dtp-logic-directory* "dtp.test"))
    (setq testfile (translate-logical-pathname testfile))
    (test-dtp-internal testfile) ))

;;;----------------------------------------------------------------------------

(defun load-logic-samples ()
  "Load the sample logical theories into DTP"
  (let (dir files file-symbol)
    (setq dir *dtp-logic-directory*)
    (setq dir (translate-logical-pathname dir))
    (setq dir (merge-pathnames "*.dtp" dir))
    (setq files (directory dir))
    (dolist (file files)
      (setq file-symbol (intern (string-upcase (pathname-name file))))
      (dolist (theory (dtp-load file))
	(unless (find theory (included-active-theory-names file-symbol))
	  (includes file-symbol theory) ))
      (includes 'logic-samples file-symbol) )))

;;;----------------------------------------------------------------------------

(defun test-tptp (&optional (problem-file nil))
  (let ((path (kif-pathname problem-file)))
    (unless problem-file
      (loop
	  initially (format t "~&Available problem files:~%  ")
	  for fl in
	    (sort (mapcar #'pathname-name (directory path)) #'string<)
	  for count from 0
	  when (= count 5)
	  do (setq count 0)
	     (format t "~%  ")
	  do (format t "~13A" fl) )
      (format t "~&File? ")
      (setq problem-file (string-downcase (read-line)))
      (setq problem-file (concatenate 'string problem-file ".kif")) )
    (setq path (merge-pathnames problem-file path))
    (unless (probe-file path)
      (format t "~&File ~A not found~%" path)
      (return-from test-tptp (values)) )
    (tptp-load path)
    (when (find :tests *trace*)
      (format t "~&Axioms:~2%")
      (show 'tptp)
      (format t "~%Proving (GOAL)~2%") )
    (let ((*theory* 'tptp))
      (prove '(goal)) )))

;;;----------------------------------------------------------------------------

(defun tptp-stats (&optional (category "PUZ"))
  "Time all TPTP problems in CATEGORY"
  (let (files time)
    (setq files (directory (tptp-pathname category)))
    (setq files (mapcar #'pathname-name files))
    (with-open-file (r *tptp-report-file*
		     :direction :output :if-exists :supersede )
      (dolist (file files)
	(format r "~A = " file)
	(setq time (gather-statistics file t))
	(if (eq time :timeout)
	    (format r "[Timeout]~%")
	  (format r "~3,1F~%" time) )))
    ))

;;;----------------------------------------------------------------------------
;;;
;;;	Private

;;;----------------------------------------------------------------------------

(defun test-dtp-internal (testfile)
  (with-open-file (tf testfile :direction :input)
    (loop
	with *package* = *dtp-package*
	with errors = 0
	with tests = 0
	initially (format t "~&")
	for sexp = (read tf nil nil)
	until (null sexp)
	for expected-answer = (read tf nil nil)
	with answer
	with result
	do (when (find :tests *trace*)
	     (if (eq expected-answer *test-ignore-answer*)
		 (format t "[~(~S~)]~%" sexp)
	       (format t "~(~S~)~%" sexp) ))
	   (setq result (multiple-value-list (eval sexp)))
	   (setq answer (first result))
	   (when (and (find :tests *trace*)
		      (not (eq expected-answer *test-ignore-answer*)) )
	     (format t "-> ~S~%" answer) )
	   (unless (eq expected-answer *test-ignore-answer*)
	     (incf tests)
	     (unless (equal expected-answer answer)
	       (incf errors)
	       (when (find :tests *trace*)
		 (format t "Error: Should have gotten~%   ~A~%"
			 expected-answer ))
	       ))
	finally
	  (when (find :tests *trace*)
	    (format t "~D test~:P checked, ~D error~:P~%" tests errors) )
	  (return errors) )))

;;;----------------------------------------------------------------------------

(defparameter *stat-frmt-str*
    (format nil "in ~~D.~~~D,'0D seconds~~%"
	    (floor (log internal-time-units-per-second 10)) ))

(defun gather-statistics (problem &optional (gc nil))
  "Problem string is DDDNNN-V[.MMM].p"
  (let (domain pn answer start-time end-time)
    (setq domain (string-upcase (subseq problem 0 3)))
    (when (string= (subseq problem (- (length problem) 2)) ".p")
      (setq problem (subseq problem 0 (- (length problem) 2))) )
    (when (find :tests *trace*) (format *debug-io* "~A: " problem))
    (when gc
      (when (find :tests *trace*) (format *debug-io* "Resetting DTP..."))
      (reset-dtp :only-internal t) )
    (setq pn (kif-pathname problem))
    (unless (probe-file pn)
      (when (find :tests *trace*) (format *debug-io* "Converting to KIF..."))
      (tptp-to-kif domain problem t nil) )
    (when (find :tests *trace*) (format *debug-io* "Loading..."))
    (tptp-load pn)
    (when gc
      (when (find :tests *trace*) (format *debug-io* "Garbage collecting..."))
      #+cmu     (ext:gc :full t)
      #+allegro (excl:gc t)
      #+lucid   (gc)
      )

    (when (find :tests *trace*) (format *debug-io* "Proving..."))
    (setq start-time (get-internal-run-time))
    (let ((*theory* 'tptp)
	  (*timeout-maximum-seconds* *tptp-timeout*) )
      (setq answer (prove '(goal))) )
    (setq end-time (get-internal-run-time))
    (when (find :tests *trace*)
      (if (equal answer '(goal))
	  (format t "Solved ")
	(format *debug-io* "Failed " answer) )
      (multiple-value-bind (sec frac)
	  (floor (- end-time start-time) internal-time-units-per-second)
	(format *debug-io* *stat-frmt-str* sec frac) ))

    (if (eq answer '(goal))
	(float (/ (- end-time start-time) internal-time-units-per-second))
      :timeout )))

;;;----------------------------------------------------------------------------
