;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;; $Source: /home/geddis/archive/cvs/scripts/dtp/lisp/literals.lisp,v $
;;; $Id: literals.lisp,v 1.1 2004/07/15 04:22:05 geddis Exp $
;;;
;;; (c) Copyright 1994-2004 Don Geddis.  All rights reserved.
;;;
;;; System:		Don's Theorem Prover (DTP)
;;; Written by:		Don Geddis <don@geddis.org>
;;;
;;;----------------------------------------------------------------------------

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Data Structures

(defstruct (literal-node
	    (:conc-name literal-)
	    (:print-function literal-node-print-function) )
  "A literal"
  negated-p
  relation
  terms )

;;;----------------------------------------------------------------------------

(defun literal-node-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<")
  (print-literal-node structure :s stream)
  (format stream ">") )

(defun print-literal-node (node &key (s t) (flip-negation nil))
  (cond
   (*print-logic-as-lists*
    (print-literal-node-as-list node :s s :flip-negation flip-negation) )
   (t
    (print-literal-node-as-logic node :s s :flip-negation flip-negation) )))

(defun print-literal-node-as-list (node &key (s t) (flip-negation nil))
  (unless (eq (literal-negated-p node) flip-negation)
    (format s "(not ") )
  (format s "(")
  (format s "~:(~A~)" (literal-relation node))
  (when (literal-terms node)
    (format s "~{ ~S~}" (literal-terms node)) )
  (format s ")")
  (unless (eq (literal-negated-p node) flip-negation)
    (format s ")") ))

(defun print-literal-node-as-logic (node &key (s t) (flip-negation nil))
  (unless (eq (literal-negated-p node) flip-negation)
    (format s "~~") )
  (format s "~:(~A~)" (literal-relation node))
  (when (literal-terms node)
    (let (term-strings)
      (setq term-strings
	(mapcar #'(lambda (term)
		    (with-output-to-string (s)
		      (term-to-string term s) ))
		(literal-terms node) ))
      (format s "(~A~{,~A~})" (car term-strings) (cdr term-strings)) )))

(defun term-to-string (term &optional (s t))
  "Variable terms -> lowercase string, Constant terms -> capitalized string"
  (cond
   ((varp term)
    (format s "~(~A~)" (variable-to-string term)) )
   ((consp term)
    (format s "~:(~A~)" (first term))
    (when (rest term) (format s "("))
    (format s "~A"
	    (with-output-to-string (str)
	      (loop
		  for remaining-terms on (rest term)
		  for subterm = (first remaining-terms)
		  do (term-to-string subterm str)
		  when (rest remaining-terms)
		  do (format str ",") )))
    (when (rest term) (format s ")")) )
   ((stringp term)
    (format s "~S" term) )
   (t
    (format s "~:(~A~)" term) )))

;;;----------------------------------------------------------------------------

(defun list-to-literal (list &optional (replacement-bindings nil))
  (let ((lit (make-literal-node)))
    (when (eq 'not (first list))
      (setf (literal-negated-p lit) t)
      (setq list (second list)) )
    (setf (literal-relation lit) (first list))
    (if replacement-bindings
	(setf (literal-terms lit) (plug (cdr list) replacement-bindings))
      (setf (literal-terms lit) (cdr list)) )
    lit ))

;;;----------------------------------------------------------------------------

(defun literal-to-list (literal &key (ignore-negation nil))
  (let ((new-list (cons (literal-relation literal) (literal-terms literal))))
    (unless ignore-negation
      (when (literal-negated-p literal)
	(setq new-list (list 'not new-list)) ))
    new-list ))

;;;----------------------------------------------------------------------------

(defun literal-list-equal-p (literal list &key (test #'equal))
  "True iff the literal is equivalent to the list representation"
  (let ((list-negation (eq (first list) 'not)))
    (when list-negation (setq list (second list)))
    (and (eq (literal-negated-p literal) list-negation)
	 (eq (literal-relation literal) (first list))
	 (funcall test (literal-terms literal) (rest list)) )))

;;;----------------------------------------------------------------------------

(defun literal-plug (literal binding-list)
  "Return a new literal, which is a copy of LITERAL with BINDING-LIST applied"
  (let ((copy (copy-literal-node literal)))
    (setf (literal-terms copy) (plug (literal-terms copy) binding-list))
    copy ))

(defun nliteral-plug (literal binding-list)
  "Destructively modify LITERAL by applying BINDING-LIST"
  (setf (literal-terms literal) (plug (literal-terms literal) binding-list))
  literal )

;;;----------------------------------------------------------------------------

(defun literal-flip-negation (literal)
  "Return a copy of LITERAL with opposite sign"
  (let ((copy (copy-literal-node literal)))
    (setf (literal-negated-p copy) (not (literal-negated-p copy)))
    copy ))

(defun nliteral-flip-negation (literal)
  "Destructively modify LITERAL to have opposite sign"
  (setf (literal-negated-p literal) (not (literal-negated-p literal)))
  literal )

;;;----------------------------------------------------------------------------

(defun query-to-answer-literal (query)
  (make-literal-node :relation 'answer_ :terms (find-vars query)) )

;;;----------------------------------------------------------------------------

(defun nliteral-rename-all-variables (literal)
  (let ((bl (literal-rename-binding-list literal)))
    (nliteral-plug literal bl) ))

(defun literal-rename-binding-list (literal)
  (mapcar
   #'(lambda (x) (cons x (make-new-variable x)))
   (remove-duplicates
    (find-vars (literal-terms literal)) )))

;;;----------------------------------------------------------------------------

(defun literal-vars-in (literal)
  "Returns list (set) of variables in terms of literal"
  (remove-duplicates (find-vars (literal-terms literal))) )

;;;----------------------------------------------------------------------------

(defun function-depth (literal)
  "The maximum depth of function application to any term in LITERAL"
  (if (literal-terms literal)
      (1- (tree-depth (literal-terms literal)))
    0 ))

;;;----------------------------------------------------------------------------
;;;
;;;	Literal equality tests

;;;----------------------------------------------------------------------------

(defun literal-possible-negated-pair-p (lit1 lit2)
  "Negated pair, but knows about EVAL, and skips terms if attachments"
  (and (not (eq (literal-negated-p lit1) (literal-negated-p lit2)))
       (eq (literal-relation lit1) (literal-relation lit2))
       (or
	*use-procedural-attachments*
	(dtp-unifyp
	 (mapcar #'eval-to-var (literal-terms lit1))
	 (mapcar #'eval-to-var (literal-terms lit2)) ))))

(defun eval-to-var (term)
  (if (and (consp term)
	   (eq (first term) 'eval) )
      (make-new-variable '?eval)
    term ))

;;;----------------------------------------------------------------------------

(defun literal-negated-pair-p (lit1 lit2 &key (test #'dtp-unifyp))
  "Returns unifying binding list, if negated pair, else nil"
  (and lit1 lit2
       (not (eq (literal-negated-p lit1) (literal-negated-p lit2)))
       (eq (literal-relation lit1) (literal-relation lit2))
       (funcall test (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun literal-mgu (lit1 lit2 &key (ignore-sign nil))
  "Returns most general unifier of terms of both literals, if exists, else nil"
  (when (and (or ignore-sign
		 (eq (literal-negated-p lit1) (literal-negated-p lit2)) )
	     (eq (literal-relation lit1) (literal-relation lit2)) )
    (dtp-unifyp (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun literal-instance
    (general-literal instance-literal &optional (old-binding-list nil))
  "True iff INSTANCE-LITERAL is an instance of GENERAL-LITERAL"
  (and (eq (literal-negated-p general-literal)
	   (literal-negated-p instance-literal) )
       (eq (literal-relation general-literal)
	   (literal-relation instance-literal) )
       (dtp-instp
	(literal-terms general-literal)
	(literal-terms instance-literal)
	old-binding-list )))

;;;----------------------------------------------------------------------------

(defun literal-instance? (instance general)
  "True IFF INSTANCE is more specific (or equal) to GENERAL"
  (literal-instance general instance) )

;;;----------------------------------------------------------------------------

(defun literal-same-or-generalized-p (instance-literal general-literal)
  "True iff I-LITERAL is same as or an instance of G-LITERAL"
  (and (eq (literal-negated-p instance-literal)
	   (literal-negated-p general-literal) )
       (eq (literal-relation instance-literal)
	   (literal-relation general-literal) )
       (dtp-instp
	(literal-terms general-literal) (literal-terms instance-literal) )))

;;;----------------------------------------------------------------------------

(defun literal-equal-p (lit1 lit2)
  (and (eq (literal-negated-p lit1) (literal-negated-p lit2))
       (eq (literal-relation lit1) (literal-relation lit2))
       (equal (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun literal-samep (lit1 lit2)
  (and (eq (literal-negated-p lit1) (literal-negated-p lit2))
       (eq (literal-relation lit1) (literal-relation lit2))
       (samep (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------
