;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;; $Source: /home/geddis/archive/cvs/scripts/dtp/lisp/figures.lisp,v $
;;; $Id: figures.lisp,v 1.1 2004/07/15 04:22:05 geddis Exp $
;;;
;;; (c) Copyright 1994-2004 Don Geddis.  All rights reserved.
;;;
;;; System:		Don's Theorem Prover (DTP)
;;; Written by:		Don Geddis <don@geddis.org>
;;;
;;;----------------------------------------------------------------------------

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(make-figures make-figure) ))

;;;----------------------------------------------------------------------------
;;;
;;;	Data

;;;----------------------------------------------------------------------------

(defparameter *figure-settings*
    '((*display-one-page* t)
      (*display-as-figure* t)
      (*display-landscape* nil)
      (*display-title* nil)
      (*display-query* nil)
      (*display-color* nil)
      (*display-cache-links* t)
      (*display-slaved-at-end?* t)
      (*display-blocked-separately* nil)
      (*print-variables-specially* :no-prefix)
      (*use-subgoal-cutoffs* nil)
      (*use-function-cutoffs* nil)
      (*caching* :postponement) ))

(defparameter *standalone-settings*
    '((*display-one-page* t)
      (*display-as-figure* nil)
      (*display-landscape* t)
      (*display-title* nil)
      (*display-query* nil)
      (*display-color* nil)
      (*display-cache-links* t)
      (*display-slaved-at-end?* t)
      (*display-blocked-separately* nil)
      (*print-variables-specially* :no-prefix)
      (*use-subgoal-cutoffs* nil)
      (*use-function-cutoffs* nil)
      (*caching* :postponement) ))

;;;----------------------------------------------------------------------------

(defparameter *thesis-figure-data*
    '(

      (answers
       ((*theory* t-answer)
	(*use-pure-literal-elimination* nil)
	(*caching* :answers) )
       (g ?x) (g ?x) )

      (backjump
       ((*theory* t-backjump))
       (g ?w) (g 6) )

      (block-1
       ((*theory* t-block)
	(*use-unblocking* nil) )
       (g) nil )
      (block-2
       ((*theory* t-block))
       (g) (g) )

      (carnivore-1
       ((*theory* t-carnivore))
       (outrun lion ?food) (outrun lion zebra) )
      (carnivore-2
       ((*theory* t-carnivore))
       (outrun lion ?food) (outrun lion dog)
       :nth-answer 2 )
      (carnivore-3
       ((*theory* t-carnivore))
       (outrun lion ?food) (outrun lion (food dog))
       :nth-answer 3 )
      (carnivore-all
       ((*theory* t-carnivore))
       (outrun lion ?food)
       ((outrun lion zebra) (outrun lion dog) (outrun lion (food dog)))
       :all-answers? t )

      ;; Warning: In DOT file, remove ~G on left, replace R on right with
      ;; "FC: C" box.
      (depth-1
       ((*theory* t-depth)
	(*caching* nil)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 4) )
       (g) (g) )
      ;; Warning: In DOT file, remove ~G on left chain.
      (depth-2
       ((*theory* t-depth)
	(*caching* nil)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 4) )
       (g) (g) )
      (depth-3
       ((*theory* t-depth)
	(*caching* nil) )
       (g) (g) )
      ;; Warning: In DOT file, remove all ~G's, replace R under C with
      ;; "FC: C" box.
      (depth-4
       ((*theory* t-depth)
	(*caching* nil)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 4) )
       (g) ((g))
       :all-answers? t )

      ;;; Warning: Must postprocess DOT file to remove links (to D and LOOKUP)
      ;;; below first B node.
      (displacement
       ((*theory* t-displacement)
	(*caching* nil)
	(*use-reordering* nil)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 4) )
       (a) (a) )

      (fact-1
       ((*theory* t-factorial))
       (f 4 24) (f 4 24)
       :show-answer? t )
      (fact-2
       ((*theory* t-factorial)
	(*caching* nil) )
       (and (i ?n) (f ?n 6)) (and (i 3) (f 3 6))
       :show-answer? t )
      (fact-3
       ((*theory* t-factorial)
	(*caching* :success) )
       (and (i ?n) (f ?n 6)) (and (i 3) (f 3 6))
       :show-answer? t )

      (fib-1
       ((*theory* t-fibonacci)
	(*caching* nil) )
       (fib 5 8) (fib 5 8)
       :show-answer? t )
      ;;; WARNING: Must postprocess.  Remove link labels, plug in numbers
      ;;; for variables.  Also, replace links by SC lookups.
      (fib-2
       ((*theory* t-fibonacci)
	(*caching* :postponement) )
	(fib 5 8) (fib 5 8) )

      (hsr-1
       ((*theory* t-hsr)
	(*caching* :iap) )
       (and (x) (y)) (and (x) (y)) )
      (hsr-2
       ((*theory* t-hsr)
	(*display-slaved-at-end?* nil)
	(*cache-reductions* nil) )
       (and (x) (y)) nil )
      (hsr-3
       ((*theory* t-hsr))
       (and (x) (y)) (and (x) (y)) )
      (hsr-b
       ((*theory* t-hsr-b)
	(*display-slaved-at-end?* nil)
	(*cache-reductions* nil) )
       (and (x) (y)) nil )

      ;;; WARNING: Must manually remove links below second C in Dot file,
      ;;; and replace with cache link to first C.  (Use iap-2 dot file.)
      (iap-1
       ((*theory* t-iap)
	(*caching* :iap) )
	(g) (g) )
      (iap-2
       ((*theory* t-iap)
	(*caching* :iap) )
       (g) (g) )

      (intuition-1
       ((*theory* t-intuition)
	(*caching* nil)
	(*use-contrapositives* nil) )
       (g) (g) )
      (intuition-2
       ((*theory* t-intuition)
	(*display-slaved-at-end?* nil)
	(*use-contrapositives* nil) )
       (g) nil )
      (intuition-3
       ((*theory* t-intuition))
       (g) (g) )

      (false-1
       ((*theory* t-false-1)
	(*caching* :iap) )
       (g) nil )
      (false-2
       ((*theory* t-false-2)
	(*caching* :iap) )
       (g) nil )
      (false-3
       ((*theory* t-false-3)
	(*caching* :iap)
	(*use-pure-literal-elimination* nil)
	(*use-backjumping* nil) )
       (g) nil )

      (flush-fc-1
       ((*theory* t-flush-fc)
	(*use-pure-literal-elimination* nil) )
       (g) nil )
      (flush-fc-2
       ((*theory* t-flush-fc)
	(*use-pure-literal-elimination* nil) )
       (not (g)) nil )
      (flush-fc-3
       ((*theory* t-flush-fc)
	(*use-pure-literal-elimination* nil)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 1) )
       (c) nil )
      (flush-fc-4
       ((*theory* t-flush-fc)
	(*use-pure-literal-elimination* nil) )
       (c) (c) )

      ;; Postprocess: Remove Path(B,end), copy twice, rename city variable
      (path-1
       ((*theory* t-path)
	(*caching* :recursion)
	(*use-reordering* nil) )
       (path a ?end) (path a b) )
      ;; Postprocess: Remove Path(B,end)
      (path-2
       ((*theory* t-path)
	(*caching* :recursion)
	(*use-reordering* nil) )
       (path a ?end) (path a b) )
      ;; Postprocess: Remove Path(C,end)
      (path-3
       ((*theory* t-path)
	(*caching* :recursion)
	(*use-reordering* nil) )
       (path a ?end) (path a c)
       :nth-answer 2 )
      (path-4
       ((*theory* t-path)
	(*caching* :recursion)
	(*use-reordering* nil) )
       (path a ?end) ((path a b) (path a c))
       :all-answers? t )
      (path-5
       ((*theory* t-path)
	(*use-reordering* nil) )
       (path a ?end) ((path a b) (path a c))
       :all-answers? t )

      (pigeon-a
       ((*theory* t-pigeon)
	(*display-landscape* t)
	(*caching* :iap) )
       (and (not (d1)) (not (d2)) (not (d3)))
       (and (not (d1)) (not (d2)) (not (d3)))
       :show-answer? t )
      (pigeon-b1
       ((*theory* t-pigeon)
	(*cache-reductions* nil)
	(*display-cache-links* :nodes) )
       (not (d1)) (not (d1)) )
      (pigeon-b2
       ((*theory* t-pigeon)
	(*cache-reductions* nil)
	(*display-cache-links* :nodes) )
       (and (not (d1)) (not (d2))) nil )
      (pigeon-c1
       ((*theory* t-pigeon)
	(*display-cache-links* :nodes) )
       (not (d1)) (not (d1)) )
      (pigeon-c2
       ((*theory* t-pigeon)
	(*display-cache-links* :nodes) )
       (and (not (d1)) (not (d2))) nil )
      (pigeon-c3
       ((*theory* t-pigeon)
	(*display-landscape* t)
	(*display-cache-links* :nodes) )
       (and (not (d1)) (not (d2)) (not (d3))) nil )

      (pure
       ((*theory* t-pure))
       (g) (g) )
      
      (sturgill
       ((*theory* t-sturgill)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 3)
	(*caching* nil) )
       (and (not (p ?x)) (d ?x)) nil )
      (sturgill-proof
       ((*theory* t-sturgill)
	(*use-subgoal-cutoffs* t)
	(*subgoal-maximum-depth* 6)
	(*caching* :recursion) )
       (and (not (p ?x)) (d ?x)) :ignore
       :show-answer? t )
      )
  "example name, var/binding pairs, query, result, &key" )

(defparameter *postprocess-figures*
    '(displacement fib-1 fib-2 iap-1 depth-1 depth-2 depth-4
		   path-1 path-2 path-3 fact-1 fact-2 fact-3 ))

(defparameter *manual-figure-data*
    '(
      (backjump
       ((*theory* m-backjump)
	(*print-variables-specially* nil) )
       (g ?w) (g 6) )
      (block-1
       ((*theory* m-block)
	(*use-unblocking* nil)
	(*print-variables-specially* nil) )
       (g) nil )
      (block-2
       ((*theory* m-block)
	(*print-variables-specially* nil) )
       (g) (g) )
      (football
       ((*theory* m-football)
	(*print-variables-specially* nil) )
       (winner ?team) (or (winner stanford) (winner cal)) )
      (mrg
       ((*theory* m-mrg)
	(*caching* nil)
	(*print-variables-specially* nil) )
       (and (p ?x ?y) (q ?x ?y)) (and (p 2 10) (q 2 10)) )
      (pure
       ((*theory* m-pure)
	(*print-variables-specially* nil) )
	(g) (g) )
      )
  "example name, var/binding pairs, query, result, &key" )

;;;----------------------------------------------------------------------------
;;;
;;;	Public

;;;----------------------------------------------------------------------------

(defun make-figures (&key (type 'thesis))
  "TYPE can be :THESIS or :MANUAL"
  (format t "Resetting DTP...~%")
  (reset-dtp)
  (case type
    ('thesis
     (dtp-load "thesis")
     (dolist (figure (mapcar #'first *thesis-figure-data*))
       (make-figure figure :type 'thesis) ))
    ('manual
     (dtp-load "manual")
     (dolist (figure (mapcar #'first *manual-figure-data*))
       (make-figure figure :type 'manual) ))
    (otherwise
     (format t ":TYPE should be 'THESIS or 'MANUAL") )))

;;;----------------------------------------------------------------------------

(defun make-figure
    (figure &key (type 'thesis) (settings *figure-settings*) (only-tmp nil))
  "TYPE can be 'THESIS or 'MANUAL"
  (let ((*figure-directory*
	 (if (eq type 'thesis)
	     *thesis-figure-directory*
	   *manual-figure-directory* ))
	(example
	 (if (eq type 'thesis)
	     (assoc figure *thesis-figure-data*)
	   (assoc figure *manual-figure-data*) ))
	(filename (format nil "~(~A~).ps" figure))
	actual-result actual-answer )
    (unless example
      (format t "Error: Can't find ~A specification for ~A~%" type figure)
      (return-from make-figure) )
    (destructuring-bind
	(name bindings query result
	      &key (show-answer? nil) (all-answers? nil) (nth-answer nil) )
	example
      (declare (ignore name))
      (format t "~A..." filename)
      (progv (mapcar #'first settings) (mapcar #'second settings)
	(progv (mapcar #'first bindings) (mapcar #'second bindings)
	  (if show-answer?
	      (multiple-value-bind (result i1 i2 answer i3)
		  (prove query
			 :all-answers all-answers? :nth-answer nth-answer )
		(declare (ignore i1 i2 i3))
		(setq actual-result result)
		(setq actual-answer answer) )
	    (setq actual-result
	      (prove query :all-answers all-answers? :nth-answer nth-answer) ))
	  (let ((*print-pretty* nil))
	    (format t "~A => ~A~%" query actual-result) )
	  (when (and (not (eq result :ignore))
		     (not (equal result actual-result)) )
	    (format t "Error: Answer should be ~A~%" result) )
	  (when (or only-tmp (find figure *postprocess-figures*))
	    (setq filename nil) )
	  (if show-answer?
	      (let ((*proof* *last-proof*))
		(make-proof-figure actual-answer filename) )
	    (make-proof-figure *last-proof* filename) )
	  ))
      )))

;;;----------------------------------------------------------------------------
