;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;; $Source: /home/geddis/archive/cvs/scripts/dtp/lisp/dotify.lisp,v $
;;; $Id: dotify.lisp,v 1.1 2004/07/15 04:22:05 geddis Exp $
;;;
;;; (c) Copyright 1994-2004 Don Geddis.  All rights reserved.
;;;
;;; System:		Don's Theorem Prover (DTP)
;;; Written by:		Don Geddis <don@geddis.org>
;;;
;;;----------------------------------------------------------------------------

(in-package "DTP")

;;;----------------------------------------------------------------------------

(defgeneric dotify (object &key parent-id answer &allow-other-keys))

;;;----------------------------------------------------------------------------

(defmethod dotify ((object proof) &key &allow-other-keys)
  (when (display-query-p object)
    (format *show-stream* "  ~A [label=~S] ;~%"
	    (dot-node-name object) (dot-name-of object) )))

;;;----------------------------------------------------------------------------

(defmethod dotify ((object answer) &key &allow-other-keys)
  (when (display-query-p (answer-proof object))
    (format *show-stream* "  ~A [label=~S] ;~%"
	    (dot-node-name object) (dot-name-of object) )))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object dtp-subgoal)
     &key (answer nil) (parent-id nil) (my-id nil) (record nil)
     (link-label nil) (previous-sg-hidden nil) (previous-sg nil)
     &allow-other-keys )
  
  (let ((my-parent nil)
	slave-link )
    
    (when answer (setq link-label nil))
    (when parent-id (setq my-parent (dot-id-to-object parent-id)))
    (unless my-id (setq my-id (dot-id-of object)))
    (setq slave-link (slave-link? object answer previous-sg))
    
    ;; Hidden subgoals
    (when (hidden-subgoal?
	   object answer parent-id record previous-sg-hidden previous-sg )
      (return-from dotify) )
    
    ;; Subgoal Node
    (when (not slave-link)
      (format *show-stream* "  ~A [shape=plaintext,label=~S] ;~%"
	      (dot-node-name my-id) (dot-name-of object answer) ))

    ;; Incoming arcs
    (when (and parent-id (or (not slave-link) *display-cache-links*))
      (when (and slave-link (eq *display-cache-links* :nodes))
	(setq my-id (dot-id-of object :unique t))
	(format *show-stream* "  ~A [shape=plaintext,fontsize=7,label=~S] ;~%"
		(dot-node-name my-id)
		(format nil "[~A]" (dot-name-of object answer)) ))
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name my-id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name my-id) ))
      (if slave-link
	  (progn
	    (cond
	     ((eq *display-cache-links* :nodes)
	      (format *show-stream* " [") )
	     ((not *display-slaved-at-end?*)
	      (format *show-stream* " [weight=1,") )
	     (t
	      (format *show-stream* " [constraint=false,") ))
	    (if *display-color*
		(format *show-stream* "color=blue")
	      (format *show-stream* "style=~A" *dash-ps-style*) ))
	(format *show-stream* " [weight=9999") )
      (if link-label
	  (format *show-stream* ",label=\"~A\"]" link-label)
	(format *show-stream* "]") )
      (format *show-stream* " ;~%") )
    ))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object dtp-conjunction)
     &key (answer nil) (parent-id nil) (my-id nil) (record nil)
     (link-label nil) &allow-other-keys )

  (when (hidden-conjunction? object)
    (return-from dotify) )
  
  (let ((name (dot-name-of object answer)))
    
    (when (and answer (eq name :lookup))
      (return-from dotify) )	; Hide LOOKUP nodes in explanations
    
    (unless my-id (setq my-id (dot-id-of object)))
    (when answer (setq link-label nil))
    
    ;; Incoming arcs
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name my-id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name my-id) ))
      (format *show-stream* " [weight=9999")
      (when link-label
	(format *show-stream* ",label=\"~A\"" link-label) )
      (format *show-stream* "] ;~%") )
    
    ;; Conjunction
    (format *show-stream* "  ~A " (dot-node-name my-id))
    (case name
      (:lookup
       (format *show-stream* "[~A,label=\"Lookup\"] ;~%"
	       *terminal-node-dot-type* ))
      (:cache
       (format *show-stream* "[~A,label=\"Cache\"] ;~%"
	       *terminal-node-dot-type* ))
      (otherwise
       (format *show-stream* "[shape=record,label=~S] ;~%" name) ))
    ))

;;;----------------------------------------------------------------------------

(defmethod dotify ((object dtp-conjunct) &key &allow-other-keys)
  "No DOT output for a conjunct"
  nil )

;;;----------------------------------------------------------------------------

(defmethod dotify ((object l-justification) &key &allow-other-keys)
  "Lookup nodes output in Dotify of Conjunction"
  nil )

;;;----------------------------------------------------------------------------

(defmethod dotify ((object c-justification) &key &allow-other-keys)
  "Conjunction justifications don't produce output"
  nil )

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object r-justification)
     &key (parent-id nil) (link-label nil) (record nil) &allow-other-keys )
  (let ((id (dot-id-of object :unique t)))
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name id) ))
      (when link-label
	(format *show-stream* " [label=\"~A\"]" link-label) )
      (format *show-stream* " ;~%") )
    (format *show-stream* "  ~A " (dot-node-name id))
    (format *show-stream* "[~A,label=\"Reduction\"] ;~%"
	    *terminal-node-dot-type* )
    ))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object res-justification)
     &key (parent-id nil) (link-label nil) (record nil) &allow-other-keys )
  (let ((id (dot-id-of object :unique t)))
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name id) ))
      (when link-label
	(format *show-stream* " [label=\"~A\"]" link-label) )
      (format *show-stream* " ;~%") )
    (format *show-stream* "  ~A " (dot-node-name id))
    (format *show-stream* "[~A,label=\"Residue\"] ;~%"
	    *terminal-node-dot-type* )))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object s-cache-justification)
     &key (parent-id nil) (link-label nil) (record nil) &allow-other-keys )
  (let ((id (dot-id-of object :unique t)))
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name id) ))
      (when link-label
	(format *show-stream* " [label=\"~A\"]" link-label) )
      (format *show-stream* " ;~%") )
    (format *show-stream* "  ~A " (dot-node-name id))
    (format *show-stream* "[~A,label=\"SC: " *terminal-node-dot-type*)
    (print-literal-node (s-cache-just-literal object) :s *show-stream*)
    (format *show-stream* "\"] ;~%") ))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object f-cache-justification)
     &key (parent-id nil) (link-label nil) (record nil) &allow-other-keys )
  (let ((id (dot-id-of object :unique t)))
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name id) ))
      (when link-label
	(format *show-stream* " [label=\"~A\"]" link-label) )
      (format *show-stream* " ;~%") )
    (format *show-stream* "  ~A " (dot-node-name id))
    (format *show-stream* "[~A,label=\"FC: " *terminal-node-dot-type*)
    (print-literal-node (f-cache-just-literal object) :s *show-stream*)
    (format *show-stream* "\"] ;~%") ))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object sg-cache-justification)
     &key (parent-id nil) (link-label nil) (record nil) &allow-other-keys )
  (let ((id (dot-id-of object :unique t)))
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name id) ))
      (when link-label
	(format *show-stream* " [label=\"~A\"]" link-label) )
      (format *show-stream* " ;~%") )
    (format *show-stream* "  ~A " (dot-node-name id))
    (format *show-stream* "[~A,label=\"SG C: " *terminal-node-dot-type*)
    (print-literal-node
     (slot-value (sg-cache-just-subgoal object) 'literal) :s *show-stream* )
    (format *show-stream* "\"] ;~%") ))

;;;----------------------------------------------------------------------------

(defmethod dotify
    ((object justification)
     &key (parent-id nil) (link-label nil) (record nil) &allow-other-keys )
  (let ((id (dot-id-of object :unique t)))
    (when parent-id
      (if record
	  (format *show-stream* "  ~A:c~D -> ~A"
		  (dot-node-name parent-id) record (dot-node-name id) )
	(format *show-stream* "  ~A -> ~A"
		(dot-node-name parent-id) (dot-node-name id) ))
      (when link-label
	(format *show-stream* " [label=\"~A\"]" link-label) )
      (format *show-stream* " ;~%") )
    (format *show-stream* "  ~A " (dot-node-name id))
    (format *show-stream* "[~A,label=\"Unknown Just\"] ;~%"
	    *terminal-node-dot-type* )))

;;;----------------------------------------------------------------------------
